const GETXHS_URL = 'https://getxhs.meltemi.fun';

// 检测当前是否在小红书页面
async function checkStatus() {
    const statusEl = document.getElementById('status');
    const extractBtn = document.getElementById('extract-btn');
    
    try {
        // 不依赖当前 tab，直接检查所有小红书 cookie
        const cookies = await chrome.cookies.getAll({ domain: '.xiaohongshu.com' });
        const hasWebSession = cookies.some(c => c.name === 'web_session');
        
        if (cookies.length === 0) {
            statusEl.className = 'status error';
            statusEl.innerHTML = '❌ 未检测到小红书 Cookie<br><small>请先在浏览器中打开并登录 xiaohongshu.com</small>';
            extractBtn.disabled = true;
            return;
        }
        
        if (hasWebSession) {
            statusEl.className = 'status ok';
            statusEl.textContent = '✅ 已检测到登录状态，可以提取！';
            extractBtn.disabled = false;
        } else {
            statusEl.className = 'status warn';
            statusEl.innerHTML = '⚠️ 检测到 Cookie 但缺少登录态<br><small>请登录小红书网页版后重试</small>';
            extractBtn.disabled = false; // 仍然允许尝试
        }
        
    } catch (e) {
        statusEl.className = 'status error';
        statusEl.textContent = '❌ ' + e.message;
        extractBtn.disabled = true;
    }
}

async function extractCookie() {
    const btn = document.getElementById('extract-btn');
    const previewEl = document.getElementById('cookie-preview');
    const statusEl = document.getElementById('status');
    
    btn.disabled = true;
    btn.textContent = '⏳ 提取中...';
    
    try {
        const cookies = await chrome.cookies.getAll({ domain: '.xiaohongshu.com' });
        
        if (cookies.length === 0) {
            statusEl.className = 'status error';
            statusEl.textContent = '❌ 未找到 Cookie，请先登录小红书';
            btn.disabled = false;
            btn.textContent = '🍪 提取 Cookie';
            return;
        }
        
        // 拼接 cookie 字符串
        const cookieStr = cookies.map(c => `${c.name}=${c.value}`).join('; ');
        
        // 优先用 Clipboard API，fallback 到 execCommand
        let copied = false;
        try {
            await navigator.clipboard.writeText(cookieStr);
            copied = true;
        } catch (_) {
            // Clipboard API 不可用时，用 textarea + execCommand fallback
            const textarea = document.createElement('textarea');
            textarea.value = cookieStr;
            textarea.style.position = 'fixed';
            textarea.style.left = '-9999px';
            document.body.appendChild(textarea);
            textarea.select();
            copied = document.execCommand('copy');
            document.body.removeChild(textarea);
        }
        
        if (copied) {
            // 显示预览
            previewEl.style.display = 'block';
            previewEl.textContent = cookieStr.substring(0, 120) + '...';
            
            statusEl.className = 'status ok';
            statusEl.textContent = '✅ Cookie 已复制到剪贴板！去网页粘贴吧';
            
            btn.textContent = '✅ 已复制！';
            btn.style.background = '#2ed573';
        } else {
            // fallback: 直接显示让用户手动复制
            previewEl.style.display = 'block';
            previewEl.textContent = cookieStr;
            previewEl.style.maxHeight = '100px';
            previewEl.style.userSelect = 'all';
            
            statusEl.className = 'status warn';
            statusEl.textContent = '⚠️ 自动复制失败，请手动选中下方文本复制';
        }
        
        setTimeout(() => {
            btn.disabled = false;
            btn.textContent = '🍪 提取 Cookie';
            btn.style.background = '';
        }, 5000);
        
    } catch (e) {
        statusEl.className = 'status error';
        statusEl.textContent = '❌ 提取失败: ' + e.message;
        btn.disabled = false;
        btn.textContent = '🍪 提取 Cookie';
    }
}

function openGetXHS() {
    // 用 window.open 作为 fallback
    try {
        chrome.tabs.create({ url: GETXHS_URL });
    } catch (e) {
        window.open(GETXHS_URL, '_blank');
    }
}

// 初始化
document.addEventListener('DOMContentLoaded', () => {
    checkStatus();
    // 绑定按钮事件（避免 inline onclick 在某些 CSP 下被拦截）
    document.getElementById('extract-btn').addEventListener('click', extractCookie);
    document.getElementById('open-btn').addEventListener('click', openGetXHS);
});
